/* notebook.c,v 1.63 2005/04/17 16:43:12 jenglish Exp
 * Copyright (c) 2004, Joe English
 *
 * TODO: Possibly: track <Map>, <Unmap> events on the master, map/unmap slaves.
 *
 * NOTE-LOSTSLAVE: Tk_ManageGeometry(slave, NULL, 0) doesn't call our 
 * LostSlaveProc() (though it probably should).  This requires extra 
 * care when removing a tab manually.
 *
 * NOTE-ACTIVE: activeTabIndex is not always correct (it's  
 * more trouble than it's worth to track this 100%)
 */

#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <tk.h>

#include "tkTheme.h"
#include "widget.h"

#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))

typedef struct NotebookStruct Notebook;	/* forward */
static void NotebookSlaveEventHandler(ClientData, XEvent *); /*forward*/
static Tk_GeomMgr NotebookGeometryManager; /* forward */

/*------------------------------------------------------------------------
 * +++ Tab resources.
 */

static const char *TabStateStrings[] = { "normal", "disabled", "hidden", 0 };
typedef enum {
    TAB_STATE_NORMAL, TAB_STATE_DISABLED, TAB_STATE_HIDDEN
} TAB_STATE;

typedef struct
{
    Notebook	*master;		/* manager widget */
    Ttk_Layout	layout;			/* Tab display layout */
    int 	width, height;		/* Requested size of tab */
    TAB_STATE 	state;			/* normal | disabled */

    /*
     * Child window options:
     */
    Tk_Window	slave;			/* slave window for this tab */
    Tcl_Obj 	*stickyObj;		/* -sticky option */
    unsigned	sticky;

    /*
     * Label resources:
     */
    Tcl_Obj *textObj;
    Tcl_Obj *imageObj;
    Tcl_Obj *compoundObj;
    Tcl_Obj *underlineObj;

} Tab;

static Tk_OptionSpec TabOptionSpecs[] =
{
    {TK_OPTION_STRING_TABLE, "-state", "", "",
	"normal", -1,Tk_Offset(Tab,state),
	0,(ClientData)TabStateStrings,0 },
    {TK_OPTION_STRING, "-sticky", "sticky", "Sticky", "nsew",
	Tk_Offset(Tab,stickyObj), -1, 0,0,GEOMETRY_CHANGED },
    {TK_OPTION_STRING, "-text", "text", "Text", "",
	Tk_Offset(Tab,textObj), -1, 0,0,GEOMETRY_CHANGED },
    {TK_OPTION_STRING, "-image", "image", "Image", NULL/*default*/,
	Tk_Offset(Tab,imageObj), -1, TK_OPTION_NULL_OK,0,GEOMETRY_CHANGED },
    {TK_OPTION_STRING_TABLE, "-compound", "compound", "Compound",
	"none", Tk_Offset(Tab,compoundObj), -1,
	0,(ClientData)TTKCompoundStrings,GEOMETRY_CHANGED },
    {TK_OPTION_INT, "-underline", "underline", "Underline", "-1",
	Tk_Offset(Tab,underlineObj), -1, 0,0,GEOMETRY_CHANGED },
    {TK_OPTION_END}
};

/*------------------------------------------------------------------------
 * +++ Notebook resources.
 */
typedef struct
{
    Tcl_Obj *widthObj;		/* Default width */
    Tcl_Obj *heightObj;		/* Default height */
    Tcl_Obj *paddingObj;	/* Internal padding */

    int nTabs;			/* #tabs */
    Tab **tabs;			/* array of tabs */
    int currentIndex;		/* index of currently selected tab */
    int activeIndex;		/* index of currently active tab */
    Tk_OptionTable tabOptionTable;

    Ttk_Box clientArea;		/* Where to pack slave widgets */
} NotebookPart;

struct NotebookStruct
{
    WidgetCore core;
    NotebookPart notebook;
};

static Tk_OptionSpec NotebookOptionSpecs[] =
{
    WIDGET_TAKES_FOCUS,

    {TK_OPTION_INT, "-width", "width", "Width", "0",
	Tk_Offset(Notebook,notebook.widthObj),-1,
	0,0,GEOMETRY_CHANGED },
    {TK_OPTION_INT, "-height", "height", "Height", "0",
	Tk_Offset(Notebook,notebook.heightObj),-1,
	0,0,GEOMETRY_CHANGED },
    {TK_OPTION_STRING, "-padding", "padding", "Padding", NULL,
	Tk_Offset(Notebook,notebook.paddingObj),-1,
	TK_OPTION_NULL_OK,0,GEOMETRY_CHANGED },

    WIDGET_INHERIT_OPTIONS(CoreOptionSpecs)
};

/*------------------------------------------------------------------------
 * +++ Tab management.
 */

static Tab *NewTab(Tcl_Interp *interp, Notebook *nb, Tk_Window slave)
{
    Tab *tab = (Tab *)ckalloc(sizeof(*tab));

    memset(tab,0,sizeof(*tab));
    if (Tk_InitOptions(interp, (ClientData)tab,
	nb->notebook.tabOptionTable, nb->core.tkwin) != TCL_OK)
    {
	ckfree((ClientData)tab);
	return 0;
    }

    tab->slave = slave;
    tab->master = nb;
    tab->layout = Ttk_CreateSubLayout(
    	interp, Ttk_GetCurrentTheme(interp), "TNotebook.Tab",
	(ClientData)tab, nb->notebook.tabOptionTable, nb->core.tkwin);

    return tab;
}

static void FreeTab(Tab *tab)
{
    Notebook *nb = tab->master;

    Tk_FreeConfigOptions((ClientData)tab,
	nb->notebook.tabOptionTable, nb->core.tkwin);
    if (tab->layout)
	Ttk_FreeLayout(tab->layout);

    ckfree((ClientData)tab);
}

static int ConfigureTab(
    Tcl_Interp *interp, Notebook *nb, Tab *tab, int objc, Tcl_Obj *CONST objv[])
{
    Tk_SavedOptions savedOptions;
    Ttk_Sticky sticky = tab->sticky;

    if (Tk_SetOptions(interp, (ClientData)tab, nb->notebook.tabOptionTable,
		objc, objv, nb->core.tkwin, &savedOptions, 0) != TCL_OK)
    {
	return TCL_ERROR;
    }

    /* Check options:
     * @@@ TODO: validate -image option with GetImageList()
     */
    if (Ttk_GetStickyFromObj(interp, tab->stickyObj, &sticky) != TCL_OK) {
	goto error;
    }

    if (tab->state == TAB_STATE_HIDDEN)
	Tk_UnmapWindow(tab->slave);

    tab->sticky = sticky;
    Tk_FreeSavedOptions(&savedOptions);
    return TCL_OK;

error:
    Tk_RestoreSavedOptions(&savedOptions);
    return TCL_ERROR;
}

/* AddTab --
 * 	Add new tab to notebook.  
 * 	Returns: index of new tab.
 */
static int AddTab(Notebook *nb, Tab *tab)
{
    int index = nb->notebook.nTabs;

    ++nb->notebook.nTabs;
    nb->notebook.tabs = (Tab**)ckrealloc(
	    (ClientData)nb->notebook.tabs, nb->notebook.nTabs * sizeof(Tab *));
    nb->notebook.tabs[index] = tab;

    /* Register the notebook as the geometry manager of the slave:
     */
    Tk_ManageGeometry(tab->slave, &NotebookGeometryManager, (ClientData)tab);

    /* Register an event handler on the slave to track destruction:
     * (??? why doesn't ManageGeometry do that for us ???)
     */
    Tk_CreateEventHandler(tab->slave, StructureNotifyMask,
	    NotebookSlaveEventHandler, (ClientData)tab);

    return index;
}

static void SelectNearestTab(Notebook *nb, int index);

static void RemoveTab(Notebook *nb, int index)
{
    Tab *tab = nb->notebook.tabs[index];
    int i;

    /* Remove tab from array:
     * It's important to do this first, see NOTE-LOSTSLAVE.
     */
    --nb->notebook.nTabs;
    for (i = index ; i < nb->notebook.nTabs ; ++i) {
	nb->notebook.tabs[i] = nb->notebook.tabs[i+1];
    }

    if (index < nb->notebook.currentIndex) {
	nb->notebook.currentIndex--;
	index = -1;
    }

    /* Free up tab resources, unmap slave window:
     */
    Tk_DeleteEventHandler(tab->slave, StructureNotifyMask,
	    NotebookSlaveEventHandler, tab);
    Tk_ManageGeometry(tab->slave, NULL, 0);	/* NOTE-LOSTSLAVE */
    Tk_UnmapWindow(tab->slave);
    FreeTab(tab);

    /* Redisplay the notebook:
     */

    WidgetChanged(&nb->core, RELAYOUT_REQUIRED);

    if (index == nb->notebook.currentIndex) {
	nb->notebook.currentIndex = -1;
	SelectNearestTab(nb, index);
    }
}

/*
 * FindTab --
 * 	Return the index of the specified slave window,
 * 	or -1 if the window is not a slave of the notebook.
 */
static int FindTab(Notebook *nb, Tk_Window slave)
{
    int index = 0;
    while (index < nb->notebook.nTabs) {
	if (nb->notebook.tabs[index]->slave == slave) {
	    return index;
	}
	++index;
    }
    return -1;
}

/*
 * IdentifyTab --
 * 	Return the index of the tab at point x,y,
 * 	or -1 if no tab at that point.
 */
static int IdentifyTab(Notebook *nb, int x, int y)
{
    int index;
    for (index = 0; index < nb->notebook.nTabs; ++index) {
	Tab *tab = nb->notebook.tabs[index];
	if (tab->state != TAB_STATE_HIDDEN) {
	    if (Ttk_LayoutIdentify(tab->layout,x,y) != NULL) {
		return index;
	    }
	}
    }
    return -1;
}

/*
 * ActivateTab --
 * 	Set the active tab index, redisplay if necessary.
 */
static void ActivateTab(Notebook *nb, int index)
{
    if (index != nb->notebook.activeIndex) {
	nb->notebook.activeIndex = index;
	WidgetChanged(&nb->core, REDISPLAY_REQUIRED);
    }
}

/*
 * TabState --
 * 	Return the state of the specified tab, based on
 * 	notebook state, currentIndex, activeIndex, and user-specified tab state.
 *	The USER1 bit is set for the leftmost tab, and USER2 
 * 	is set for the rightmost tab.  USER3 bit is set for hidden tabs.
 */
static Ttk_State TabState(Notebook *nb, int index)
{
    Ttk_State state = nb->core.state;

    if (index == nb->notebook.currentIndex) {
	state |= TTK_STATE_SELECTED;
    } else {
	state &= ~TTK_STATE_FOCUS;
    }

    if (index == nb->notebook.activeIndex) {
	state |= TTK_STATE_ACTIVE;
    }
    if (index == 0) {
    	state |= TTK_STATE_USER1;
    }
    if (index == nb->notebook.nTabs - 1) {
    	state |= TTK_STATE_USER2;
    }
    if (nb->notebook.tabs[index]->state == TAB_STATE_DISABLED) {
	state |= TTK_STATE_DISABLED;
    }

    if (nb->notebook.tabs[index]->state == TAB_STATE_HIDDEN) {
	state |= TTK_STATE_USER3;
    }

    return state;
}


/*------------------------------------------------------------------------
 * +++ Geometry management.
 */

/* SqueezeTabs --
 *	If the notebook is not wide enough to display all tabs,
 *	attempt to decrease tab widths to fit.
 *
 *	All tabs are shrunk by an equal amount, but will not be made 
 *	smaller than the minimum width.  (If all the tabs still do
 *	not fit in the available space, the rightmost tabs are truncated).
 *
 *	The algorithm does not always yield an optimal layout, but does
 *	have the important property that decreasing the available width
 *	by one pixel will cause at most one tab to shrink by one pixel;
 *	this means that tabs resize "smoothly" when the window shrinks
 *	and grows.
 */
static void SqueezeTabs(
    Notebook *nb, int desiredWidth, int availableWidth, int minTabWidth)
{
    int nTabs = nb->notebook.nTabs;
    int shrinkage = desiredWidth - availableWidth;
    int extra = 0;
    int i;

    for (i = 0; i < nTabs; ++i) {
	Tab *tab = nb->notebook.tabs[i];
	int shrink = (shrinkage/nTabs) + (i < (shrinkage%nTabs)) + extra;
	int shrinkability = MAX(0, tab->width - minTabWidth);
	int delta = MIN(shrinkability, shrink);
	tab->width -= delta;
	extra = shrink - delta;
    }
}

/* NotebookDoLayout --
 *	Computes notebook layout and places tabs.
 *
 * Side effects:
 * 	Sets clientArea, used to place slave panes.
 */
static void NotebookDoLayout(void *recordPtr)
{
    Notebook *nb = recordPtr;
    Tk_Window nbwin = nb->core.tkwin;
    Ttk_Box cavity = Ttk_MakeBox(0,0, Tk_Width(nbwin),Tk_Height(nbwin));
    Ttk_Padding expandTab = Ttk_UniformPadding(0);
    Tcl_Obj *expandTabObj = Ttk_QueryOption(nb->core.layout, "-expandtab", 0);
    int width = 0, height = 0;
    Ttk_LayoutNode *clientNode = Ttk_LayoutFindNode(nb->core.layout, "client");
    Ttk_Box tabrowBox;
    int i;

    /* Notebook internal padding:
     */
    if (nb->notebook.paddingObj) {
	Ttk_Padding padding;
	Ttk_GetPaddingFromObj(
	    NULL,nb->core.tkwin,nb->notebook.paddingObj,&padding);
	cavity = Ttk_PadBox(cavity, padding);
    }

    /* Extra space for selected tab:
     */
    if (expandTabObj) {
	Ttk_GetPaddingFromObj(NULL, nb->core.tkwin, expandTabObj, &expandTab);
    }

    /* Layout for notebook background (base layout):
     */
    Ttk_PlaceLayout(nb->core.layout, nb->core.state, Ttk_WinBox(nbwin));

    /* Compute max height and total width of all tabs:
     */
    for (i = 0; i<nb->notebook.nTabs; ++i) {
	Tab *tab = nb->notebook.tabs[i];
	Ttk_State tabState = TabState(nb,i);
	if (!(tabState & TTK_STATE_USER3)) {
	    Ttk_LayoutSize(tab->layout,tabState,&tab->width,&tab->height);
	    if (tab->height > height) {
	        height = tab->height;
	    }
	    width += tab->width;
	}
    }

    /*
     * If there are no visible tabs, height will still be zero.  Calculate
     * the height of a dummy tab and use it to prevent redrawing problems
     * such as clipping and movement of the bar that appears between the
     * tabs and page body under some themes.
     */

    if (height == 0) {
	Tab *dummy = NewTab(nb->core.interp, nb, NULL);
	if (dummy) {
	    Ttk_LayoutSize(dummy->layout,nb->core.state,&dummy->width,&height);
	    FreeTab(dummy);
	}
    }

    /* Place tabs:
     * Allow extra space on the top, left, and right to account for expandTab.
     */
    tabrowBox = Ttk_PackBox(&cavity, width, height+expandTab.top, TTK_SIDE_TOP);
    tabrowBox.x += expandTab.left;
    tabrowBox.width -= Ttk_PaddingWidth(expandTab);

    if (width > tabrowBox.width) {
	int minTabWidth = height;
	Tcl_Obj *minTabWidthObj = 
	    Ttk_QueryOption(nb->core.layout, "-mintabwidth", 0);
	if (minTabWidthObj) {
	    (void)Tcl_GetIntFromObj(NULL, minTabWidthObj, &minTabWidth);
	}
	SqueezeTabs(nb, width, tabrowBox.width, minTabWidth);
    }

    for (i = 0; i<nb->notebook.nTabs; ++i) {
	Tab *tab = nb->notebook.tabs[i];
	unsigned tabState = TabState(nb, i);
	if (!(tabState & TTK_STATE_USER3)) {
	    Ttk_Box tabBox = Ttk_PlaceBox(&tabrowBox, tab->width,
		tab->height, TTK_SIDE_LEFT, TTK_STICK_S);
	    if (tabState & TTK_STATE_SELECTED) {
	        tabBox = Ttk_ExpandBox(tabBox, expandTab);
	    }
	    Ttk_PlaceLayout(tab->layout, tabState, tabBox);
	}
    }

    /* Layout for client area frame:
     */
    if (clientNode) {
	Ttk_LayoutNodeSetParcel(clientNode, cavity);
	cavity = Ttk_LayoutNodeInternalParcel(nb->core.layout, clientNode);
    }

    if (cavity.height <= 0) cavity.height = 1;
    if (cavity.width <= 0) cavity.width = 1;

    nb->notebook.clientArea = cavity;
}

/*
 * NotebookPlaceSlave --
 * 	Set the position and size of a child widget
 * 	based on the current client area and slave's -sticky option:
 */
static void NotebookPlaceSlave(Tab *tab)
{
    Notebook *nb = tab->master;
    int slaveWidth = Tk_ReqWidth(tab->slave);
    int slaveHeight = Tk_ReqHeight(tab->slave);
    Ttk_Box slaveBox = Ttk_StickBox(
	nb->notebook.clientArea, slaveWidth, slaveHeight, tab->sticky);

    Tk_MoveResizeWindow(	/* %%%NOTE-CHILD */
	tab->slave, slaveBox.x, slaveBox.y, slaveBox.width, slaveBox.height);
}

/*
 * Notebook geometry manager procedures:
 */
static void
NotebookGeometryRequestProc(ClientData clientData, Tk_Window slave)
{
    Tab *tab = clientData;
    WidgetChanged(&tab->master->core, RELAYOUT_REQUIRED);
}

static void
NotebookGeometryLostSlaveProc(ClientData clientData, Tk_Window slave)
{
    Tab *tab = (Tab *)clientData;
    Notebook *nb = tab->master;
    int index;

    /* Locate slave in tab array, remove entry:
     */
    for (index=0; index<nb->notebook.nTabs; ++index) {
	if (nb->notebook.tabs[index] == tab) {
	    RemoveTab(nb, index);
	    return;
	}
    }
}

static Tk_GeomMgr NotebookGeometryManager = {
    "notebook",
    NotebookGeometryRequestProc,
    NotebookGeometryLostSlaveProc
};

/*------------------------------------------------------------------------
 * +++ Event handlers.
 */

/* NotebookSlaveEventHandler --
 * 	Notifies the master when a slave is destroyed.
 */
static void
NotebookSlaveEventHandler(ClientData clientData, XEvent *eventPtr)
{
    Tab *tab = (Tab*)clientData;
    if (eventPtr->type == DestroyNotify) {
	NotebookGeometryLostSlaveProc(clientData, tab->slave);
    }
}

/* NotebookEventHandler --
 * 	Track <Configure> events, resize active child.
 * 	Also tracks the active tab (we can't use TrackElementState()
 * 	for this, since we manage tab layouts ourselves).
 */
static const int NotebookEventMask 
    = StructureNotifyMask 
    | PointerMotionMask
    | LeaveWindowMask 
    ;
static void NotebookEventHandler(ClientData clientData, XEvent *eventPtr)
{
    Notebook *nb = (Notebook*)clientData;

    if (eventPtr->type == DestroyNotify) {
	/* Remove self */
	Tk_DeleteEventHandler(nb->core.tkwin, NotebookEventMask,
		NotebookEventHandler, clientData);
    } else if (eventPtr->type == ConfigureNotify) {
	/* Re-pack active slave: */
	NotebookDoLayout(nb);
	if (nb->notebook.currentIndex >= 0) {
	    NotebookPlaceSlave(nb->notebook.tabs[nb->notebook.currentIndex]);
	}
    } else if (eventPtr->type == MotionNotify) {
	int index = IdentifyTab(nb, eventPtr->xmotion.x, eventPtr->xmotion.y);
	ActivateTab(nb, index);
    } else if (eventPtr->type == LeaveNotify) {
	ActivateTab(nb, -1);
    }
}

/*------------------------------------------------------------------------
 * +++ Utilities.
 */

/* GetTabIndex --
 *	Find the index of the specified tab.
 *	Tab identifiers are one of:
 *
 *	+ numeric indices [0..nTabs],
 *	+ slave window names
 *	+ positional specifications @x,y
 *	+ "current"
 *
 *	Returns: index of specified tab, -1 if not found.
 *	If 'interp' is non-NULL, leaves an error message there.
 */
static int GetTabIndex(
    Tcl_Interp *interp,		/* Where to leave error message if non-NULL */
    Notebook *nb,		/* Notebook widget record */
    Tcl_Obj *objPtr,		/* Tab name to look up */
    int *index_rtn)
{
    const char *string = Tcl_GetString(objPtr);
    int index, x, y;

    *index_rtn = -1;

    /*
     * Check for integer indices:
     */
    if (isdigit(string[0]) && Tcl_GetIntFromObj(NULL,objPtr,&index)==TCL_OK) {
	if (index >= 0 && index < nb->notebook.nTabs) {
	    *index_rtn = index;
	    return TCL_OK;
	} else {
	    if (interp)
		Tcl_SetResult(interp, "Index out of range", TCL_STATIC);
	    return TCL_ERROR;
	}
    }

    /*
     * Or window path names...
     */
    if (string[0] == '.') {
	for (index = 0; index < nb->notebook.nTabs; ++index) {
	    Tab *tab = nb->notebook.tabs[index];
	    if (!strcmp(string, Tk_PathName(tab->slave))) {
		*index_rtn = index;
		return TCL_OK;
	    }
	}
	if (interp) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, string,
		    " is not managed by ", Tk_PathName(nb->core.tkwin),
		    NULL);
	}
	return TCL_ERROR;
    }

    /*
     * Or @x,y ...
     */
    if (string[0] == '@' && sscanf(string, "@%d,%d",&x,&y) == 2) {
	*index_rtn = IdentifyTab(nb, x, y);
	return TCL_OK;
    }

    /*
     * Or "current"
     */
    if (!strcmp(string, "current")) {
	*index_rtn = nb->notebook.currentIndex;
	return TCL_OK;
    }

    /*
     * Nothing matches.
     */
    if (interp) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "Invalid tab index ", string, NULL);
    }

    return TCL_ERROR;
}

/*
 * SelectTab --
 * 	Select the specified tab.
 */
static void SelectTab(Notebook *nb, int index)
{
    Tab *tab = nb->notebook.tabs[index], *oldTab = 0;
    int currentIndex = nb->notebook.currentIndex;

    if (index == currentIndex) {
	return;
    }

    if (tab->state == TAB_STATE_DISABLED) {
	return;
    }

    /* Unhide the tab if it is currently hidden and being selected. */

    if (tab->state == TAB_STATE_HIDDEN) {
	tab->state = TAB_STATE_NORMAL;
    }

    if (currentIndex >= 0) {
	oldTab = nb->notebook.tabs[currentIndex];
	Tk_UnmapWindow(oldTab->slave);
    }

    NotebookPlaceSlave(tab);
    Tk_MapWindow(tab->slave);

    nb->notebook.currentIndex = index;
    WidgetChanged(&nb->core, REDISPLAY_REQUIRED);

    SendVirtualEvent(nb->core.tkwin, "NotebookTabChanged");
}

/*
 * SelectNearestTab --
 * 	Choose the next closest tab to the specified tab that is in
 *	the normal state.  This is called to handle WidgetChanged
 *	notifications for the current tab during configuration or
 *	tab deletion.  If the current tab is deleted or set to hidden
 *	or disabled state, the next closest tab (if any) is selected.
 */
static void SelectNearestTab(Notebook *nb, int index)
{
    int newIndex = -1;

    for ( ; index < nb->notebook.nTabs ; index++) {
	if (nb->notebook.tabs[index]->state == TAB_STATE_NORMAL) {
	    newIndex = index;
	    break;
	}
    }

    if (newIndex == -1) {
	for (index = nb->notebook.nTabs - 1 ; index >= 0 ; index--) {
	    if (nb->notebook.tabs[index]->state == TAB_STATE_NORMAL) {
		newIndex = index;
		break;
	    }
	}
    }

    if (newIndex != -1) {
	SelectTab(nb, newIndex);
    } else {
	nb->notebook.currentIndex = -1;
	WidgetChanged(&nb->core, RELAYOUT_REQUIRED);
    }
}

/*------------------------------------------------------------------------
 * +++ Widget command routines.
 */

/*
 * $nb add window [ options ... ]
 */
static int NotebookAddCommand(
    Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[], void *recordPtr)
{
    Notebook *nb = recordPtr;
    Tk_Window slave;
    Tab *tab = 0;
    int index;

    if (objc <= 2 || objc % 2 != 1) {
	Tcl_WrongNumArgs(interp, 2, objv, "window ?options...?");
	return TCL_ERROR;
    }

    slave = Tk_NameToWindow(interp, Tcl_GetString(objv[2]), nb->core.tkwin);
    if (!slave) {
	return TCL_ERROR;
    }

    /*
     * Only allow direct children of the notebook to be added as slaves.
     * (%%%NOTE-CHILD-RESTRICTION might relax this later)
     */
    if (Tk_Parent(slave) != nb->core.tkwin) {
	Tcl_AppendResult(interp,
	    "can't add ", Tk_PathName(slave),
	    " to ", Tk_PathName(nb->core.tkwin),
	    NULL);
	return TCL_ERROR;
    }

    /*
     * Make sure slave is not already present:
     */
    if (FindTab(nb, slave) != -1) {
	Tcl_AppendResult(interp, Tk_PathName(slave), " already added", NULL);
	return TCL_ERROR;
    }

    /* Create and initialize new tab:
     */
    tab = NewTab(interp, nb, slave);
    if (!tab) {
	return TCL_ERROR;
    }
    if (ConfigureTab(interp, nb, tab, objc-3, objv+3) != TCL_OK) {
	FreeTab(tab);
	return TCL_ERROR;
    }

    index = AddTab(nb, tab);

    /* If no tab is currently selected (or if this is the first tab),
     * select this one:
     */
    if (nb->notebook.currentIndex < 0) {
	SelectTab(nb, index);
    }

    WidgetChanged(&nb->core, REDISPLAY_REQUIRED|RELAYOUT_REQUIRED);

    return TCL_OK;
}

/*
 * $nb forget $item --
 * 	Removes the selected tab.
 */
static int NotebookForgetCommand(
    Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[], void *recordPtr)
{
    Notebook *nb = recordPtr;
    int index, status;

    if (objc != 3) {
	Tcl_WrongNumArgs(interp, 2, objv, "tab");
	return TCL_ERROR;
    }

    status = GetTabIndex(interp, nb, objv[2], &index);
    if (status == TCL_OK) {
	RemoveTab(nb, index);	/* NOTE-LOSTSLAVE */
    }

    return status;
}

/*
 * $nb index $item --
 * 	Returns the integer index of the tab specified by $item,
 * 	the empty string if $item does not identify a tab.
 *	See above for valid item formats.
 */
static int NotebookIndexCommand(
    Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[], void *recordPtr)
{
    Notebook *nb = recordPtr;
    int index, status;

    if (objc != 3) {
	Tcl_WrongNumArgs(interp, 2, objv, "tab");
	return TCL_ERROR;
    }

    /*
     * Special-case for "end":
     */
    if (!strcmp("end", Tcl_GetString(objv[2]))) {
	Tcl_SetObjResult(interp, Tcl_NewIntObj(nb->notebook.nTabs));
	return TCL_OK;
    }

    status = GetTabIndex(interp, nb, objv[2], &index);
    if (status == TCL_OK && index >= 0) {
	Tcl_SetObjResult(interp, Tcl_NewIntObj(index));
    }

    return status;
}

/*
 * $nb select $item --
 * 	Selects the specified tab.
 */
static int NotebookSelectCommand(
    Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[], void *recordPtr)
{
    Notebook *nb = recordPtr;
    int index, status;

    if (objc != 3) {
	Tcl_WrongNumArgs(interp, 2, objv, "tab");
	return TCL_ERROR;
    }

    status = GetTabIndex(interp, nb, objv[2], &index);
    if (status == TCL_OK) {
	SelectTab(nb, index);
    }

    return status;
}

/*
 * $nb tabs --
 * 	Return list of tabs.
 */
static int NotebookTabsCommand(
    Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[], void *recordPtr)
{
    Notebook *nb = recordPtr;
    Tcl_Obj *result;
    int i;

    if (objc != 2) {
	Tcl_WrongNumArgs(interp, 2, objv, "");
	return TCL_ERROR;
    }

    result = Tcl_NewListObj(0, NULL);

    for (i = 0; i < nb->notebook.nTabs; ++i) {
	Tcl_ListObjAppendElement(interp, result,
	    Tcl_NewStringObj(Tk_PathName(nb->notebook.tabs[i]->slave), -1));
    }

    Tcl_SetObjResult(interp, result);
    return TCL_OK;
}

/* $nb tab $tab ?-option ?value -option value...??
 */
static int NotebookTabCommand(
    Tcl_Interp *interp, int objc, Tcl_Obj * CONST objv[], void *recordPtr)
{
    Notebook *nb = recordPtr;
    int index;
    Tab *tab;

    if (objc < 3) {
	Tcl_WrongNumArgs(interp, 2, objv, "tab ?-option ?value??...");
	return TCL_ERROR;
    }

    if (GetTabIndex(interp, nb, objv[2], &index) != TCL_OK) {
	return TCL_ERROR;
    }
    tab = nb->notebook.tabs[index];

    if (objc == 3) {
	return EnumerateOptions(interp, tab, TabOptionSpecs,
	    nb->notebook.tabOptionTable, nb->core.tkwin);
    } else if (objc == 4) {
	return GetOptionValue(interp, tab, objv[3],
	    nb->notebook.tabOptionTable, nb->core.tkwin);
    } else {
	int status = ConfigureTab(interp, nb, tab, objc - 3,objv + 3);
	if (status == TCL_OK) {
	    WidgetChanged(&nb->core, RELAYOUT_REQUIRED);

	    if ((index == nb->notebook.currentIndex) ||
		    (nb->notebook.currentIndex < 0))
		SelectNearestTab(nb, index);
	}
	return status;
    }
}

/* Subcommand table:
 */
static WidgetCommandSpec NotebookCommands[] =
{
    { "add",    	NotebookAddCommand },
    { "configure",	WidgetConfigureCommand },
    { "cget",		WidgetCgetCommand },
    { "forget",		NotebookForgetCommand },
    { "index",		NotebookIndexCommand },
    { "instate",	WidgetInstateCommand },
    { "select",		NotebookSelectCommand },
    { "state",  	WidgetStateCommand },
    { "tab",   		NotebookTabCommand },
    { "tabs",   	NotebookTabsCommand },
    { 0,0 }
};

/*------------------------------------------------------------------------
 * +++ Widget class hooks.
 */

static int NotebookInitialize(Tcl_Interp *interp, void *recordPtr)
{
    Notebook *nb = recordPtr;

    nb->notebook.nTabs = 0;
    nb->notebook.tabs = NULL;
    nb->notebook.currentIndex = -1;
    nb->notebook.activeIndex = -1;
    nb->notebook.tabOptionTable = Tk_CreateOptionTable(interp,TabOptionSpecs);

    nb->notebook.clientArea = Ttk_MakeBox(0,0,1,1);

    Tk_CreateEventHandler(
	nb->core.tkwin, NotebookEventMask, NotebookEventHandler, recordPtr);

    return TCL_OK;
}

static void NotebookCleanup(void *recordPtr)
{
    Notebook *nb = recordPtr;
    if (nb->notebook.tabs) {
	int i;
	for (i=0; i<nb->notebook.nTabs; ++i)
	    FreeTab(nb->notebook.tabs[i]);
	ckfree((ClientData)nb->notebook.tabs);
    }
}

static int NotebookConfigure(Tcl_Interp *interp, void *clientData, int mask)
{
    Notebook *nb = clientData;

    /*
     * Error-checks:
     */
    if (nb->notebook.paddingObj) {
	/* Check for valid -padding: */
	/* @@@ Make this a CustomOptionType */
	Ttk_Padding unused;
	if (Ttk_GetPaddingFromObj(
		    interp, nb->core.tkwin, nb->notebook.paddingObj, &unused)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
    }

    return CoreConfigure(interp, clientData, mask);
}

/* NotebookGetLayout  --
 * @@@ TODO: error checks
 */
static Ttk_Layout NotebookGetLayout(
	Tcl_Interp *interp, Ttk_Theme theme, void *recordPtr)
{
    Notebook *nb = recordPtr;
    int i;

    for (i=0; i<nb->notebook.nTabs; ++i) {
	Tab *tab = nb->notebook.tabs[i];
	Ttk_FreeLayout(tab->layout);
	tab->layout = Ttk_CreateSubLayout(
	    interp, theme, "TNotebook.Tab", 
	    tab, nb->notebook.tabOptionTable, nb->core.tkwin);
    }

    return Ttk_CreateLayout(interp, theme, "TNotebook", 
	    recordPtr, nb->core.optionTable, nb->core.tkwin);
}


/* NotebookSize --
 * 	Client area determined by max size of slaves,
 * 	overridden by -width and/or -height if nonzero.
 *
 * Total height is tab height + client area height + pane internal padding
 * Total width is max(client width, tab width) + pane internal padding
 */

static int NotebookSize(void *clientData, int *widthPtr, int *heightPtr)
{
    Notebook *nb = clientData;
    Ttk_Padding padding = Ttk_UniformPadding(0);
    Tcl_Obj *expandTabObj = Ttk_QueryOption(nb->core.layout, "-expandtab", 0);
    Ttk_LayoutNode *clientNode = Ttk_LayoutFindNode(nb->core.layout, "client");
    int clientWidth = 0, clientHeight = 0,
    	reqWidth = 0, reqHeight = 0,
	tabrowWidth = 0, tabrowHeight = 0;
    int i;

    /* Compute max requested size of all slaves:
     */
    for (i = 0; i < nb->notebook.nTabs; ++i) {
	Tk_Window slave = nb->notebook.tabs[i]->slave;
	clientWidth = MAX(clientWidth, Tk_ReqWidth(slave));
	clientHeight = MAX(clientHeight, Tk_ReqHeight(slave));
    }

    /* Client width/height overridable by resources:
     */
    Tcl_GetIntFromObj(NULL, nb->notebook.widthObj,&reqWidth);
    Tcl_GetIntFromObj(NULL, nb->notebook.heightObj,&reqHeight);
    if (reqWidth > 0)
	clientWidth = reqWidth;
    if (reqHeight > 0)
	clientHeight = reqHeight;

    /* Compute max height and total width of all tabs:
     * @@@FACTOR THIS
     */
    for (i = 0; i<nb->notebook.nTabs; ++i) {
	Tab *tab = nb->notebook.tabs[i];
	Ttk_LayoutSize(tab->layout,TabState(nb,i),&tab->width,&tab->height);
	tabrowHeight = MAX(tabrowHeight, tab->height);
	tabrowWidth += tab->width;
    }
    if (expandTabObj) {
	Ttk_Padding expandTab;
	Ttk_GetPaddingFromObj(NULL, nb->core.tkwin, expandTabObj, &expandTab);
	tabrowHeight += expandTab.top;
    }

    /* Account for exterior padding:
     */
    if (nb->notebook.paddingObj) {
	Ttk_GetPaddingFromObj(
		NULL, nb->core.tkwin, nb->notebook.paddingObj, &padding);
    }

    /* And interior padding:
     */
    if (clientNode) {
	Ttk_Padding ipad =
	    Ttk_LayoutNodeInternalPadding(nb->core.layout, clientNode);
	padding = Ttk_AddPadding(padding, ipad);
    }

    *widthPtr = MAX(tabrowWidth, clientWidth) + Ttk_PaddingWidth(padding);
    *heightPtr = tabrowHeight + clientHeight + Ttk_PaddingHeight(padding);

    return 1;
}

static void DisplayTab(Notebook *nb, int index, Drawable d)
{
    Tab *tab = nb->notebook.tabs[index];
    Ttk_State state = TabState(nb, index);

    if (tab->state != TAB_STATE_HIDDEN) {
	Ttk_DrawLayout(tab->layout, state, d);
    }
}

static void NotebookDisplay(void *clientData, Drawable d)
{
    Notebook *nb = clientData;
    int index;

    /* Draw notebook background (base layout):
     */
    Ttk_DrawLayout(nb->core.layout, nb->core.state, d);

    /* Draw tabs from left to right, but draw the current tab last
     * so it will overwrite its neighbors. 
     */
    for (index = 0; index < nb->notebook.nTabs; ++index) {
	if (index != nb->notebook.currentIndex) {
	    DisplayTab(nb, index, d);
	}
    }
    if (nb->notebook.currentIndex >= 0) {
	DisplayTab(nb, nb->notebook.currentIndex, d);
    }
}

/*------------------------------------------------------------------------
 * +++ Widget specification and layout definitions.
 */

static WidgetSpec NotebookWidgetSpec =
{
    "TNotebook",		/* className */
    sizeof(Notebook),		/* recordSize */
    NotebookOptionSpecs,	/* optionSpecs */
    NotebookCommands,		/* subcommands */
    NotebookInitialize,		/* initializeProc */
    NotebookCleanup,		/* cleanupProc */
    NotebookConfigure,		/* configureProc */
    NullPostConfigure,		/* postConfigureProc */
    NotebookGetLayout, 		/* getLayoutProc */
    NotebookSize,		/* geometryProc */
    NotebookDoLayout,		/* layoutProc */
    NotebookDisplay,		/* displayProc */
    WIDGET_SPEC_END		/* sentinel */
};

TTK_BEGIN_LAYOUT(NotebookLayout)
    TTK_NODE("Notebook.client", TTK_FILL_BOTH)
TTK_END_LAYOUT

TTK_BEGIN_LAYOUT(TabLayout)
    TTK_GROUP("Notebook.tab", TTK_FILL_BOTH,
	TTK_GROUP("Notebook.padding", TTK_PACK_TOP|TTK_FILL_BOTH, 
	    TTK_GROUP("Notebook.focus", TTK_PACK_TOP|TTK_FILL_BOTH, 
		TTK_NODE("Notebook.label", TTK_PACK_TOP))))
TTK_END_LAYOUT

int Notebook_Init(Tcl_Interp *interp)
{
    Ttk_Theme themePtr = Ttk_GetDefaultTheme(interp);

    Ttk_RegisterLayout(themePtr, "Tab", TabLayout);
    Ttk_RegisterLayout(themePtr, "TNotebook", NotebookLayout);

    RegisterWidget(interp, "ttk::notebook", &NotebookWidgetSpec);

    return TCL_OK;
}

/*EOF*/
