/* trace.c,v 1.3 2005/03/26 20:02:07 jenglish Exp
 *
 * Copyright 2003, Joe English
 *
 * Simplified interface to Tcl_TraceVariable.
 */

#include <tk.h>
#include "tkTheme.h"
#include "widget.h"

struct TraceHandle_
{
    Tcl_Interp	*interp;
    Tcl_Obj 	*varnameObj;		/* Name of variable being traced */
    TraceProc	callback;		/* Callback procedure */
    void	*clientData;		/* Data to pass to callback */
};

static char *
VarTraceProc(
    ClientData clientData,	/* Widget record pointer */
    Tcl_Interp *interp, 	/* Interpreter containing variable. */
    CONST char *name1,		/* (unused) */
    CONST char *name2,		/* (unused) */
    int flags)			/* Information about what happened. */
{
    TraceHandle *tracePtr = (TraceHandle*)clientData;
    const char *name, *value;
    Tcl_Obj *valuePtr;

    if (flags & TCL_INTERP_DESTROYED)
	return NULL;

    name = Tcl_GetString(tracePtr->varnameObj);

    /*
     * If the variable is being unset, then re-establish the trace:
     */
    if (flags & TCL_TRACE_DESTROYED) {
	Tcl_TraceVar(interp, name,
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		VarTraceProc, clientData);

	(void)tracePtr->callback(interp, tracePtr->clientData, NULL);

	return NULL;
    }

    /*
     * Call the callback:
     */
    valuePtr = Tcl_GetVar2Ex(interp, name, NULL, TCL_GLOBAL_ONLY);
    if (valuePtr == NULL) {
        value = NULL;
    } else {
        value = Tcl_GetString(valuePtr);
    }

    /* @@@ todo: deal with errors */
    (void)tracePtr->callback(interp, tracePtr->clientData, value);

    return NULL;
}

/* TraceVariable(interp, varNameObj, callback, clientdata) --
 * 	Attach a write trace to the specified variable,
 * 	which will call 'callback' with the variable's value
 * 	whenever the variable is set.
 *
 * 	When the variable is unset, passes NULL to the callback
 * 	and reattach the trace.
 */ 	
TraceHandle *TraceVariable(
    Tcl_Interp *interp,
    Tcl_Obj *varnameObj,
    TraceProc callback,
    void *clientData)
{
    TraceHandle *h = (TraceHandle*)ckalloc(sizeof(TraceHandle));
    int status;

    h->interp = interp;
    h->varnameObj = Tcl_DuplicateObj(varnameObj);
    Tcl_IncrRefCount(h->varnameObj);
    h->clientData = clientData;
    h->callback = callback;

    status = Tcl_TraceVar(interp, Tcl_GetString(varnameObj),
	    TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
	    VarTraceProc, (ClientData)h);

    if (status != TCL_OK) {
	Tcl_DecrRefCount(h->varnameObj);
	ckfree((char*)h);
	return NULL;
    }

    return h;
}

/*
 * UntraceVariable(TraceHandle *h) --
 * 	Remove previously-registered trace handle.
 */
void UntraceVariable(TraceHandle *h)
{
    if (!h) return;
    Tcl_UntraceVar(h->interp, Tcl_GetString(h->varnameObj),
	    TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
	    VarTraceProc, (ClientData)h);
    Tcl_DecrRefCount(h->varnameObj);
    ckfree((char*)h);
}

/*
 * FireTrace --
 * 	Executes a trace handle as if the variable has been written.
 * 	Note: may reenter the interpreter.
 */
int FireTrace(TraceHandle *tracePtr)
{
    Tcl_Interp *interp = tracePtr->interp;
    WidgetCore *corePtr = tracePtr->clientData;
    const char *name = Tcl_GetString(tracePtr->varnameObj);
    TraceProc callback = tracePtr->callback;
    Tcl_Obj *valuePtr;
    const char *value;

    /*
     * Read the variable.
     * Note that this can reenter the interpreter, and anything can happen --
     * including the current trace handle being freed!
     */
    valuePtr = Tcl_GetVar2Ex(interp, name, NULL, TCL_GLOBAL_ONLY);
    value = valuePtr ? Tcl_GetString(valuePtr) : NULL;

    if (WidgetDestroyed(corePtr))
	return TCL_ERROR;

    return callback(interp, corePtr, value);
}
/*EOF*/
