# Setup a simple status bar using the sizegrip element to get the correct
# graphical element in the bottom corner of the application window.
#
# Simple because you can only set a single label that fills the whole bar.
# 

package require Tk
package require tile

# Configure the style
#
style theme settings default {

    style layout TStatusbar {
        Statusbar.background
        Statusbar.border -children {
            Statusbar.padding -children {
                Statusbar.label -side left -sticky nsw
            }
        }
        Statusbar.sizegrip -side right -sticky se
    }
    style configure TStatusbar -padding {2 1} -relief sunken ;#-font {Tahoma 8}
}

# Create a widget command to create the simple statusbar
#
proc ttk::statusbar {name args} {
    return [eval [linsert $args 0 \
                      ttk::label $name -anchor w \
                      -style TStatusbar -class TStatusbar]]
}

# Event handling and cursor manipulation.
#

namespace eval ttk::statusbar {
    variable State
    bind TStatusbar <Motion>          { ttk::statusbar::Motion %W %x %y }
    bind TStatusbar <ButtonPress-1>   { ttk::statusbar::Press %W %x %y %X %Y }
    bind TStatusbar <B1-Motion>       { ttk::statusbar::Drag %W %X %Y }
    bind TStatusbar <ButtonRelease-1> { ttk::statusbar::Release %W %X %Y }
}

proc ttk::statusbar::Motion {W x y} {
    variable State
    switch -exact -- [$W identify $x $y] {
        sizegrip {
            if {![info exists State(cursor)]} {
                set State(cursor) [$W cget -cursor]
                $W configure -cursor size_nw_se
            }
        }
        default {
            if {[info exists State(cursor)]} {
                $W configure -cursor $State(cursor)
                unset State(cursor)
            }
        }
    }
}

proc ttk::statusbar::Press {W x y X Y} {
    variable State

    if {![string equal "sizegrip" [$W identify $x $y]]} {
        return
    }

    set top [winfo toplevel $W]
    catch { 
	::sizegrip $W $X $Y
	event generate . <Motion>	;# See <<NOTE-MOUSEBUTTONSTATE>>
    }

    # Sanity-checks:
    #	If a negative X or Y position was specified for [wm geometry],
    #   just bail out -- there's no way to handle this cleanly.
    #
    if {[scan [wm geometry .] "%dx%d+%d+%d" width height _x _y] != 4} {
	return;
    }

    # Handle gridded geometry
    #
    set grid [wm grid $top]
    if {[llength $grid]} {
	set State(widthInc) [lindex $grid 2]
	set State(heightInc) [lindex $grid 3]
    } else {
	set State(widthInc) [set State(heightInc) 1]
    }

    set State(toplevel) $top
    set State(pressX) $X
    set State(pressY) $Y
    set State(width)  $width
    set State(height) $height

    set State(pressed) 1
}

proc ttk::statusbar::Drag {W X Y} {
    variable State
    if {!$State(pressed)} { return }
    set w [expr {$State(width) + ($X - $State(pressX))/$State(widthInc)}]
    set h [expr {$State(height) + ($Y - $State(pressY))/$State(heightInc)}]
    if {$w <= 0} { set w 1 }
    if {$h <= 0} { set h 1 }
    wm geometry $State(toplevel) ${w}x${h}
}

proc ttk::statusbar::Release {W X Y} {
    variable State
    set State(pressed) 0
}
